using System;
using System.Data;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using Method = System.Reflection.MethodBase;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for OrderReport.
	/// </summary>
	public class OrderReport:VbecsReport
	{
		private Guid _orderGuid;
		private int _recordCount = int.MinValue;
        private DataSet _dsReportData = new DataSet();

		#region constructors

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5480"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Instance of OrderReport</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5481"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OrderReport
		/// </summary>
		public OrderReport()
		{
			// multi order report by default
			IsSingleOrderReport = false;
			SortByTreatingSpecialty = true;
		}
		
		#endregion


		#region properties

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5482"> 
		///		<ExpectedInput>None (single order)</ExpectedInput>
		///		<ExpectedOutput>Count > 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5484"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the number of records
		/// </summary>
		public int RecordCount
		{
			get
			{
				if (this.IsSingleOrderReport)
				{
					// They have selected an order.  This is always 1!
					return 1;
				}
				else
				{
					_recordCount = DAL.OrderReport.GetMultiOrderReportCount(this.StartDate, this.EndDate, Common.LogonUser.LogonUserDivisionCode);
				}
				
				return _recordCount;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5485"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5486"> 
		///		<ExpectedInput>true (with OrderGuid = Empty)</ExpectedInput>
		///		<ExpectedOutput>Broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether this is a single or multi-order report
		/// </summary>
		public bool IsSingleOrderReport
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsSingleOrderReport))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsSingleOrderReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsSingleOrderReport] = value;
				if (IsSingleOrderReport)
				{
					RuleBroken("OrderGuid", _orderGuid == Guid.Empty);
				}
				else
				{
					RuleBroken("OrderGuid", false);
				}
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5488"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5489"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsComponentOrder
		/// </summary>
		public bool IsComponentOrder
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.IsComponentOrder))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IsComponentOrder]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IsComponentOrder] = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5490"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5491"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specifies sorting in multiple order reports only
		/// </summary>
		public bool SortByTreatingSpecialty
		{
			get
			{
				if (this.ParameterRow.IsNull(ARTIFICIAL.SortByTreatingSpecialty))
				{
					return false;
				}
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.SortByTreatingSpecialty]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.SortByTreatingSpecialty] = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5492"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="5494"> 
		///		<ExpectedInput>Empty Guid (with SingleOrderReport = true)</ExpectedInput>
		///		<ExpectedOutput>Broken rule</ExpectedOutput>
		///	</Case>
		///	
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specifies order Guid (component or test) in a single order report only
		/// </summary>
		public Guid OrderGuid
		{
			get
			{
				return _orderGuid;
			}
			set
			{
				_orderGuid = value;
				RuleBroken(Method.GetCurrentMethod().Name, _orderGuid == Guid.Empty && IsSingleOrderReport == true);
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5507"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-empty string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5508"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SearchCriteria
		/// </summary>
		public string SearchCriteria
		{
			get
			{
				if (IsSingleOrderReport)
				{
					return "Search by patient";
				}
				else
				{
					return "Order received between " + Common.VBECSDateTime.FormatDateString(StartDate) + " and " + Common.VBECSDateTime.FormatDateString(EndDate) + 
						", sorted by " + (this.SortByTreatingSpecialty ? "treating specialty." : "ordering location.") ;
				}
			}
		}

		#endregion


		#region methods

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5495"> 
		///		<ExpectedInput>No broken rules</ExpectedInput>
		///		<ExpectedOutput>DataSet representing the report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5496"> 
		///		<ExpectedInput>Broken rule count > 0</ExpectedInput>
		///		<ExpectedOutput>Business object layer exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataSet representing the report
		/// </summary>
		/// <returns></returns>
		public DataSet CreateReport()
		{
			if (this.BrokenRulesCount > 0)
			{
				throw new BOL.BusinessObjectException("There are broken rules, so the report cannot be created.");
			}

			DataSet dsOrderHistory = new DataSet();

			if (IsSingleOrderReport)
			{
				dsOrderHistory.DataSetName = "SingleOrderHistoryReport";
				//
				DataTable dtOrder = new DataTable();
				//
				DataTable dtSpecimenHistory = new DataTable();
				DataTable dtReflexTests = new DataTable();
				//
				bool specimenHistoryExists = false;
				bool reflexTestsExist = false;

				// Main order
				if (IsComponentOrder)
				{
					dtOrder = DAL.OrderReport.GetReportSingleOrderComponentHistory(_orderGuid);
				}
				else
				{
					dtOrder = DAL.OrderReport.GetReportSingleOrderTestHistory(_orderGuid);
				}

				dtOrder.TableName = "Order";
				dsOrderHistory.Tables.Add(dtOrder.Copy());
			
				// Specimen history
				Guid specimenGuid = dtOrder.Rows[0].IsNull(TABLE.PatientSpecimen.PatientSpecimenGuid) ? Guid.Empty : (Guid) dtOrder.Rows[0][TABLE.PatientSpecimen.PatientSpecimenGuid];
				if (specimenGuid != Guid.Empty)
				{
					dtSpecimenHistory = BOL.Specimen.GetSpecimenHistory(specimenGuid).Copy();
					if (dtSpecimenHistory.Rows.Count > 0)
					{
						dtSpecimenHistory.TableName = "SpecimenHistory";
						specimenHistoryExists = true;
						dsOrderHistory.Tables.Add(dtSpecimenHistory);
					}
				}

				// Reflex tests
				dtReflexTests = BOL.PatientOrder.GetReflexTestsForOrder(_orderGuid);
				if (dtReflexTests.Rows.Count > 0)
				{
					dtReflexTests.TableName = "ReflexTests";
					reflexTestsExist = true;
					dsOrderHistory.Tables.Add(dtReflexTests.Copy());
				}

				// Modify Order table
				dsOrderHistory.Tables[0].Columns.Add(ARTIFICIAL.SpecimenHistoryExists,typeof(bool));
				dsOrderHistory.Tables[0].Columns.Add(ARTIFICIAL.ReflexTestsExist,typeof(bool));
				dsOrderHistory.Tables[0].Columns.Add(ARTIFICIAL.IsComponentOrder,typeof(bool));
				dsOrderHistory.Tables[0].Columns.Add(ARTIFICIAL.OrderExpirationDate,typeof(DateTime));
				dsOrderHistory.Tables[0].Rows[0][ARTIFICIAL.SpecimenHistoryExists] = specimenHistoryExists;
				dsOrderHistory.Tables[0].Rows[0][ARTIFICIAL.ReflexTestsExist] = reflexTestsExist;
				dsOrderHistory.Tables[0].Rows[0][ARTIFICIAL.IsComponentOrder] = IsComponentOrder;
				if (specimenGuid != Guid.Empty)
				{
					dsOrderHistory.Tables[0].Rows[0][ARTIFICIAL.OrderExpirationDate] = (DateTime) dtOrder.Rows[0][TABLE.PatientSpecimen.SpecimenExpirationDate];
				}
				else
				{
					dsOrderHistory.Tables[0].Rows[0][ARTIFICIAL.OrderExpirationDate] = ((DateTime) dtOrder.Rows[0][TABLE.PatientOrder.OrderReceivedDatetime]).AddDays(10);
				}

				// Modify SpecimenHistory table
				if (specimenHistoryExists)
				{
					dsOrderHistory.Tables[1].Columns.Add(TABLE.BloodUnitMedia.Comments);
					foreach (DataRow dtRow in dsOrderHistory.Tables[1].Rows )
					{
						dtRow[TABLE.BloodUnitMedia.Comments] = string.Empty;
					}
				}
#if DEBUG
				//dsOrderHistory.WriteXml(@"C:\dataset.xml", System.Data.XmlWriteMode.WriteSchema);
#endif
			}
			else
			{
				dsOrderHistory.DataSetName = "MultiOrderHistoryReport";
				DataTable dtOrder = DAL.OrderReport.GetReportMultiOrderHistory(this.StartDate, this.EndDate, Common.LogonUser.LogonUserDivisionCode);
				dtOrder.TableName = "Order";
				dsOrderHistory.Tables.Add(dtOrder.Copy());
			}
			return dsOrderHistory;
		}

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6123"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6124"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate and print report
		/// Used by VbecsConsoleReport application
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            _dsReportData = this.CreateReport();
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            if (this.IsSingleOrderReport)
            {
                DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);

                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReportData.Tables.Count > 0)
                {
                    reportSourceMain.Name = "SingleOrderDataSet";
                    reportSourceMain.Value = _dsReportData.Tables["Order"];
                }

                ReportDataSource reportSourceHeader = new ReportDataSource();
                if (dsHeader.Tables.Count > 0)
                {
                    reportSourceHeader.Name = "ReportHeaderDataSet";
                    reportSourceHeader.Value = dsHeader.Tables[0];
                }
                Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
                Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.OrderHistory.SingleOrderHistoryReport.rdlc");
                Stream streamSpecimen = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.OrderHistory.SpecimenHistory.rdlc");
                Stream streamReflexTests = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.OrderHistory.ReflexTests.rdlc");

                LocalReport report = new LocalReport();
                report.SubreportProcessing += new SubreportProcessingEventHandler(LoadSubreportsEventHandler);

                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(stream);
                report.LoadSubreportDefinition("SpecimenHistory", streamSpecimen);
                report.LoadSubreportDefinition("ReflexTests", streamReflexTests);

                return new BOL.VbecsReportPrinter(this.PrinterName, false, report);
            }
            else
            {
                DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
                DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.SearchCriteria);

                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReportData.Tables.Count > 0)
                {
                    reportSourceMain.Name = "MultiOrderDataSet";
                    reportSourceMain.Value = _dsReportData.Tables[0];
                }

                ReportDataSource reportSourceHeader = new ReportDataSource();
                if (dsHeader.Tables.Count > 0)
                {
                    reportSourceHeader.Name = "ReportHeaderDataSet";
                    reportSourceHeader.Value = dsHeader.Tables[0];
                }

                ReportDataSource reportSourceFooter = new ReportDataSource();
                if (dsFooter.Tables.Count > 0)
                {
                    reportSourceFooter.Name = "ReportFooterDataSet";
                    reportSourceFooter.Value = dsFooter.Tables[0];
                }

                Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
                Stream stream;

                if (this.SortByTreatingSpecialty)
                {
                    stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.OrderHistory.MultiOrderTreatingSpecialty.rdlc");
                }
                else
                {
                    stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.OrderHistory.MultiOrderOrderingLocation.rdlc");
                }

                LocalReport report = new LocalReport();
                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceFooter);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(stream);

                return new BOL.VbecsReportPrinter(this.PrinterName, false, report);
            }
		}

        private void LoadSubreportsEventHandler(object sender, SubreportProcessingEventArgs e)
        {
            if (e.ReportPath == "SpecimenHistory" || e.DataSourceNames[0].ToString() == "SpecimenDataSet")
            {
                ReportDataSource reportSpecimen = new ReportDataSource();
                if (_dsReportData.Tables.IndexOf("SpecimenHistory") != -1)
                {
                    reportSpecimen.Name = "SpecimenDataSet";
                    reportSpecimen.Value = _dsReportData.Tables["SpecimenHistory"];
                }
                e.DataSources.Add(reportSpecimen);
            }
            else if (e.ReportPath == "ReflexTests" || e.DataSourceNames[0].ToString() == "ReflexTestsDataSet")
            {
                ReportDataSource reportReflexTests = new ReportDataSource();
                if (_dsReportData.Tables.IndexOf("ReflexTests") != -1)
                {
                    reportReflexTests.Name = "ReflexTestsDataSet";
                    reportReflexTests.Value = _dsReportData.Tables["ReflexTests"];
                }
                e.DataSources.Add(reportReflexTests);
            }
        }

		#endregion

	}
}
